// ----------------------------------------------------------------------------
// SPON_deathMessages
//
// Version: v0.2.1
// Copyright (C) 2008 Bil Bas (bil {dot} bagpuss [at] gmail {dot} com / Spooner)
// License: GNU Lesser General Public License, version 3 <http://www.gnu.org/licenses/>
//
// Description:
//   Shows messages when someone dies. This is only useful if death messages are
//   turned off on the server, but it does give greater control over who gets
//   the messages and when they receive them.
//
//   If BIS default deathMessages are enabled on the server, then this script
//   does nothing.
//
//   Example, run in init.sqf with:
//     [false, true, false, true, false, true, [resistance], ["commanderNoddy", "commanderBigEars"], 60] execVM "SPON_deathMesssages.sqf";
//
// Limitations:
//   * Being run over, intentionally or otherwise, is an "unexplained death".
//   * Death due to spawned artillery rounds (or similar) will show as an
//     "unexplained death".
//   * Never shows AI deaths (although it does show players killed by AI).
//
// Parameters:
//   0: _showEnemyKills - Show all kills where it is obvious who killed who [Boolean].
//   1: _showTeamkills - Show all teamkills [Boolean].
//   2: _showTeamkillsToEnemy = Show teamkills to people on other sides than the corpse [Boolean]
//   3: _showUnexplainedDeaths - Show all exexplained deaths (respawn, fall, etc) [Boolean].
//   4: _showUnexplainedDeathsToEnemy - show unexplained deaths to people on other sides than the corpse [Boolean].
//   5: _leadersNotified - Group leaders are notified [Boolean].
//   6: _sidesNotified - Notify anyone in one of these sides. If array is empty,
//       [], notify no-one based on side [Array of Sides].
//   7: _individualsNotified - Individuals to be notified, regardless of their
//        side. Note that this is the STRING var-name, not the object itself.
//       The names are case-dependent! [Array of Strings].
//   8: _delay - Delay before informing of the death (seconds). Delay is local,
//       so can be different for different players [Number: >= 0].
//
// Changelog:
//   v0.2.1
//     FIXED: Messages are still seen on a SPON deathMessages enabled server [Reported by Dr Eyeball].
//
// ----------------------------------------------------------------------------

#define UNEXPLAINED_DEATH 0
#define TEAM_KILL 1
#define ENEMY_KILL 2

// Name uses for all AI that kill players.
#define AI_NAME "AI"

// For shorter delays, shows it in seconds. For this many or more, shows minutes.
#define MIN_DELAY_SHOW_MINUTES 120

// ----------------------------------------------------------------------------

if (isServer and (isNull player)) exitWith {}; // Don't run on dedicated server.

if (difficultyEnabled "deathMessages") exitWith {}; // Don't duplicate messages.

private ["_sidesNotified", "_individualsNotified"];
SPON_DM_showEnemyKills = _this select 0;
SPON_DM_showTeamkills = _this select 1;
SPON_DM_showTeamkillsToEnemy = _this select 2;
SPON_DM_showUnexplainedDeaths = _this select 3;
SPON_DM_showUnexplainedDeathsToEnemy = _this select 4;
SPON_DM_leadersNotified = _this select 5;
SPON_DM_sidesNotified = _this select 6;
SPON_DM_individualsNotified = _this select 7;
SPON_DM_delay = _this select 8;

// Ensure that player exists to ensure that it exists (so we can check whether
// it can be notified of messages) and to ensure that JIP syncing has occurred
// (so he doesn't get "out-of-date" messages).
waitUntil { alive player };

// Remember player's name, since using "name _corpse" produces an RPT error.
SPON_DM_playerName = name player;
SPON_DM_system = "Logic" createVehicle [0, 0, 0];

SPON_DM_showMessage =
{
	private ["_victim", "_killer", "_type", "_victimSide"];
	_victim = _this select 0;
	_killer = _this select 1;
	_type = _this select 2;
	_victimSide = _this select 3;
	
	// Check whether the player is entitled to get death messages at all.
	if (not ((SPON_DM_leadersNotified and (player == (leader player)) or
		(playerSide in SPON_DM_sidesNotified) or
		((vehicleVarName player) in SPON_DM_individualsNotified)))) exitWith {};
		
	// Check whether teamkill messages should be hidden from the player.
	if ((_type == TEAM_KILL) and (not SPON_DM_showTeamkillsToEnemy) and
		(playerSide != _victimSide)) exitWith {};
		
	// Check whether unexplained messages should be hidden from the player.
	if ((_type == UNEXPLAINED_DEATH) and (not SPON_DM_showUnexplainedDeathsToEnemy) and
		(playerSide != _victimSide)) exitWith {};
	
	private "_message";
	
	switch _type do
	{
		case UNEXPLAINED_DEATH:
		{
			_message = format ["%1 died", _victim];
		};
		case TEAM_KILL:
		{
			_message = format ["%1 killed by %2 (Friendly fire)", _victim, _killer];
		};
		case ENEMY_KILL:
		{
			_message = format ["%1 killed by %2", _victim, _killer];
		};
	};
	
	if (SPON_DM_delay == 0) then
	{
		SPON_DM_system globalChat _message;
	}
	else
	{
		[_message] spawn
		{
			private "_message";
			_message = _this select 0;
			
			sleep SPON_DM_delay;
			
			_message = if (SPON_DM_delay >= MIN_DELAY_SHOW_MINUTES) then
			{
				format ["%1 about %2 minutes ago", _message, round (SPON_DM_delay / 60)];
			}
			else
			{
				format ["%1 %2 seconds ago", _message, ceil SPON_DM_delay];
			};
			
			SPON_DM_system globalChat _message;
		};
	};
};

// Redirect event to handler.
"SPON_DM_deathEvent" addPublicVariableEventHandler
{
	(_this select 1) call SPON_DM_showMessage;
};

// When the local player is killed, broadcast this event.
player addEventHandler ["KILLED",
{
	private ["_victim", "_killer", "_type"];
	_victim = _this select 0;
	_killer = _this select 1;
	
	if ((isNull _killer) or (_victim == _killer)) then
	{
		if (SPON_DM_showUnexplainedDeaths) then
		{
			// Unexpected death - collision, respawn or fall.
			_type = UNEXPLAINED_DEATH;
		};
	}
	else{if (playerSide == (side _killer)) then
	{
		if (SPON_DM_showTeamkills) then
		{
			// Definite team-kill.
			_type = TEAM_KILL;
		};
	}
	else
	{
		if (SPON_DM_showEnemyKills) then
		{
			_type = ENEMY_KILL;
		};
	}; };
	
	if (not (isNil "_type")) then
	{
		private ["_params", "_killerName"];
		
		_killerName = if (isNull _killer) then
		{
			"";
		}
		else{if (isPlayer _killer) then
		{
			name _killer;
		}
		else
		{
			AI_NAME;
		}; };
		
		_params = [SPON_DM_playerName, _killerName, _type, playerSide];
		
		// Tell the local player, if they have messages enabled.
		if (not (isNil "SPON_DM_showMessage")) then
		{
			_params call SPON_DM_showMessage;
		};
		
		// Tell anyone else.
		SPON_DM_deathEvent = _params;
		publicVariable "SPON_DM_deathEvent";
	};
	
	nil; // Return.
}];

nil; // Return.
