// SPON Rangefinder v0.3.0 (Mission-script version)
//
// Copyright (C) 2007-8 Bil Bas (bil {dot} bagpuss [at] gmail {dot} com / Spooner)
// License: GNU Lesser General Public License, version 3 <http://www.gnu.org/licenses/>

// -----------------------------------------------------------------------------
#define SPON_THIS_FILE SPON\Rangefinder\initRangefinder
//
// Description:
//   Add a rangefinder overlay to the laser designator.
//
// Parameters:
//   0: _possibleMilsPer360Degrees - Number of mils in 360 degrees
//        (for azimuth & elevation). The first value will be the one initially set.
//        Set to [] to hide angle values entirely.
//        [Array of number, defaults to [6000, 6283, 6400]]
//   1: _laserDesignatorClass - Name of the laser-designator to make a
//        SPON Rangefinder [String, defaults to "LaserDesignator"]
//   2: _showMapMarker - Whether to show an icon on the map when friendly
//        rangefinder is being used (still requires rangefinder-user to have
//        GPS before this position is shown) [Boolean, defaults to false]
//
// Returns:
//   nil
//
// -----------------------------------------------------------------------------

#include "macros\macros.inc.sqf"

#include "rangefinder.inc.sqf"

// Needed for the idd/idcs of the rangefinder.
#include "ui\handles.hpp"

// -----------------------------------------------------------------------------
// Ensure that Rangefinder isn't initialised more than once.
if (not (isNil "SPON_Rangefinder_version")) exitWith
{
	private "_version";
	_version = '$RELEASE_VERSION$';
	SPON_TRACE_2("SPON Rangefinder run when already initialised",SPON_Rangefinder_version,_version);
	
	nil; // Return.
};

SPON_Rangefinder_version = '$RELEASE_VERSION$';

// -----------------------------------------------------------------------------
// The internal handler for dealing with finding a laser marker does some stuff
// before it passes the information on to the user.
private "_targetFoundHandler";
_targetFoundHandler =
{
	SPON_GET_PARAMS_4(_target,_owner,_targetPos,_gps);
	
	SPON_TRACE_4(TARGET_FOUND_EVENT_PRIVATE,_target,_owner,_targetPos,_gps);
	
	if ((not (isNull _target)) and (not (isNull _owner))) then
	{
		_target setVariable [LASER_TARGET_OWNER_VAR, _owner];
		
		// Keep the SPON_RF_targets list up to date.
		SPON_PUSH(SPON_RF_targets,_target);
		[_target] spawn
		{
			SPON_GET_PARAMS_1(_target);
			waitUntil { isNull _target };
			SPON_RF_targets = SPON_RF_targets - [_target];
		};
		
		// Show map marker, if necessary.
		if (SPON_RF_mapMarkerEnabled and _gps and SPON_isClient) then
		{
			// Show target marker to players on the same side.
			if (playerSide == (side _owner)) then
			{
				[_target, name _owner] spawn
				{
					SPON_GET_PARAMS_2(_target,_ownerName);

					if (not (isNull _target)) then
					{
						_marker = createMarkerLocal [format [TARGET_MARKER, SPON_RF_targetNum],
							getPos _target];
						SPON_RF_targetNum = SPON_RF_targetNum + 1;
						
						_marker setMarkerTypeLocal TARGET_ICON;
						_marker setMarkerTextLocal (format
							[localize "STR_SPON_RF_LASER_MARKER_LABEL",
								_ownerName]);
						_marker setMarkerSizeLocal TARGET_ICON_SIZE;
						
						while { not (isNull _target) } do
						{
							_marker setMarkerPosLocal (getPos _target);
							
							sleep 0.5;
						};
						
						deleteMarkerLocal _marker;
					};
				};
			};
		};
	};
	
	[TARGET_FOUND_EVENT_PUBLIC, _this] call SPON_publishLocalEvent;	
};

// -----------------------------------------------------------------------------

SPON_RF_setMilsPer360Degrees =
{
	SPON_GET_PARAMS_1(_milsPer360Degrees);
	
	if (not (_milsPer360Degrees in SPON_RF_possibleMilsPer360Degrees)) exitWith
	{
		SPON_ASSERT_OP(_milsPer360Degrees,in,SPON_RF_possibleMilsPer360Degrees,"SPON_RF_setMilsPer360Degrees requires a valid value");
	};
	
	if (not isNull SPON_RF_display) then
	{
		private "_control";
		
		_control = SPON_RF_display displayCtrl SPON_RF_MODE_LIST_IDC;
		
		// Turn off previous setting.
		_control lbSetPicture [
			SPON_RF_possibleMilsPer360Degrees find SPON_RF_milsPer360Degrees,
			SPON_RF_LED_OFF];
		
		// Turn on new setting.
		_control lbSetPicture [
			SPON_RF_possibleMilsPer360Degrees find _milsPer360Degrees,
			SPON_RF_LED_ON];
	};
	
	SPON_RF_milsPer360Degrees = _milsPer360Degrees;
	
	SPON_TRACE_1("SPON_RF_setMilsPer360Degrees",SPON_RF_milsPer360Degrees);
};

// -----------------------------------------------------------------------------

SPON_RF_fillModeList =
{
	private ["_list", "_index"];
	_modeListBox = SPON_RF_display displayCtrl SPON_RF_MODE_LIST_IDC;
	
	lbClear _modeListBox;
	
	{
		if (_x == 360) then
		{
			_modeListBox lbAdd (localize "STR_SPON_RF_MODE_DEGREES");
		}
		else
		{
			_modeListBox lbAdd format [localize "STR_SPON_RF_MODE_MILS", _x];
		};
		
		_index = (lbSize _modeListBox) - 1;
		
		_modeListBox lbSetValue [_index, _x];
		
		if (_x == SPON_RF_milsPer360Degrees) then
		{
			_modeListBox lbSetPicture [_index, SPON_RF_LED_ON];
		}
		else
		{
			_modeListBox lbSetPicture [_index, SPON_RF_LED_OFF];
		};
		
		SPON_RF_possibleMilsPer360Degrees
	} forEach SPON_RF_possibleMilsPer360Degrees;
	
	nil; // Return.
};

// -----------------------------------------------------------------------------

SPON_RF_setLaserDesignatorClass =
{
	SPON_GET_PARAMS_1(_laserDesignatorClass);
	
	SPON_RF_laserDesignatorClass = _laserDesignatorClass;
	
	nil; // Return.
};

// -----------------------------------------------------------------------------

SPON_RF_enableMapMarker =
{
	SPON_GET_PARAMS_1(_showMapMarker);
	
	SPON_RF_mapMarkerEnabled = _showMapMarker;

	nil; // Return.
};

// -----------------------------------------------------------------------------
SPON_RF_possibleMilsPer360Degrees = DEFAULT_POSSIBLE_MILS_PER_REVOLUTION;

[DEFAULT_LD_CLASS] call SPON_RF_setLaserDesignatorClass;
[DEFAULT_SHOW_MAP_MARKER] call SPON_RF_enableMapMarker;

if ((count SPON_RF_possibleMilsPer360Degrees) > 0) then
{
	SPON_RF_milsPer360Degrees = SPON_RF_possibleMilsPer360Degrees select 0;
}
else
{
	SPON_RF_milsPer360Degrees = 0;
};

SPON_RF_targets = [];

SPON_RF_actionsOn = nullObj; // Settings action is present on what player object.
SPON_RF_actionIndices = []; // Indices of each action added.
SPON_RF_targetNum = 0; // To ensure we get a unique marker name each time.
SPON_RF_display = controlNull; // Will be set by onLoad.

SPON_TRACE_4("SPON Rangefinder initiated",SPON_RF_possibleMilsPer360Degrees,SPON_RF_milsPer360Degrees,SPON_RF_laserDesignatorClass,SPON_RF_mapMarkerEnabled);

// Detect the global (private) event.
[TARGET_FOUND_EVENT_PRIVATE, _targetFoundHandler] call SPON_addEventHandler;

if (SPON_isDedicatedServer) exitWith { nil }; // Display monitoring is client-side.

SPON_findLaserTarget = compile preprocessFileLineNumbers "SPON\Rangefinder\findLaserTarget.sqf";

SPON_RF_settingsKeyDown = false;

private "_settingsKeyDownHandler";
_settingsKeyDownHandler =
{
	if ((alive player) and (not (isNull SPON_RF_Display)) and (not SPON_RF_settingsKeyDown)) then
	{
		// Move to next available mode.
		_index = SPON_RF_possibleMilsPer360Degrees find SPON_RF_milsPer360Degrees;
		_index = (_index + 1) mod (count SPON_RF_possibleMilsPer360Degrees);
		[SPON_RF_possibleMilsPer360Degrees select _index] call SPON_RF_setMilsPer360Degrees;
		
		SPON_RF_settingsKeyDown = true;
		
		true; // Return.
	}
	else
	{
		// Allow other weapons to reload normally.
		false; // Return.
	};
};

{
	["DOWN", _x, _settingsKeyDownHandler] call SPON_addKeyEventHandler;
} forEach (actionKeys "ReloadMagazine");

private "_settingsKeyUpHandler";
_settingsKeyUpHandler =
{
	if (not (isNull SPON_RF_Display)) then
	{
		SPON_RF_settingsKeyDown = false;
		
		true; // Return.
	}
	else
	{
		// Allow other weapons to reload normally.
		false; // Return.
	};
};

{
	["UP", _x, _settingsKeyUpHandler] call SPON_addKeyEventHandler;
} forEach (actionKeys "ReloadMagazine");


[] spawn
{
	private ["_target"];
		
	_target = objNull;
	
	// Check whether the player has a range-finder (slow poll)
	while {true} do
	{
		// Check whether the player is using the range-finder (fast poll).
		if (player hasWeapon SPON_RF_laserDesignatorClass) then
		{
			waitUntil // not (player hasWeapon SPON_RF_laserDesignatorClass);
			{
				if ((animationState player) in ANIM_BINOCS) then
				{
					if (SPON_RF_milsPer360Degrees > 0) then
					{
						call SPON_RF_addMilsActions;
					};
					
					private ["_rangeControl", "_azimuthControl",
						"_elevationControl", "_laserOnControl",
						"_laserBatteriesControl", "_batteryTypes", "_hasBattery"];
					
					waitUntil // not ((animationState player) in ANIM_BINOCS);
					{
						// Make sure the display is always shown when looking through
						// laser designator.
						if (isNull SPON_RF_Display) then
						{
							cutRsc ["SPON_RF_overlay", "PLAIN", 0];
							
							// Can't do anything until we have found the GUI element.
							waitUntil { not (isNull SPON_RF_display); };
							
							call SPON_RF_fillModeList;
							
							_rangeControl = SPON_RF_display displayCtrl
								SPON_RF_RANGE_IDC;
							_azimuthControl = SPON_RF_display displayCtrl
								SPON_RF_AZIMUTH_IDC;
							_elevationControl = SPON_RF_display displayCtrl
								SPON_RF_ELEVATION_IDC;
							_laserOnControl = SPON_RF_display displayCtrl
								SPON_RF_LASER_ON_IDC;
							_laserBatteriesControl = SPON_RF_display displayCtrl
								SPON_RF_BATTERY_ON_IDC;
							
							_rangeControl ctrlSetText VALUE_NOT_SHOWN;
							
							// Laser designator is being used, but the laser is
							// initially off unless the overlay was closed by
							// another script while laser was on.
							if (not (isNull _target)) then
							{
								_laserOnControl ctrlSetText SPON_RF_LED_ON;
							};
							
							// Remove some of the overlay if using the NWD improved optics addon.
							if (NWD_OPTICS_INSTALLED()) then
							{
								SPON_TRACE("NWD ScopeFix detected. Making display compatible.");
								
								{
									(SPON_RF_display displayCtrl _x) ctrlShow false;
								} forEach SPON_RF_BACKGROUND_IDCS;
							};
						};
						
						// If we haven't already detected a target, try to find one to
						// see if the laser is being fired.		
						if (isNull _target) then
						{
							_target = call SPON_findLaserTarget;
							
							if (isNull _target) then
							{
								_laserOnControl ctrlSetText SPON_RF_LED_OFF;
							}
							else
							{
								// Make sure that everyone knows who the target belongs to.
								// We send the position, in case the target has been destroyed
								// (designator turned off) before everyone picks up the event.
								[TARGET_FOUND_EVENT_PRIVATE,
									[_target, player, getPos _target, shownGPS]] call
										SPON_publishGlobalEvent;
								
								_laserOnControl ctrlSetText SPON_RF_LED_ON;
							};
						};
						
						// Show battery light if player has a battery in inventory.
						_batteryTypes = getArray (configFile >> "CfgWeapons" >>
							SPON_RF_laserDesignatorClass >> "magazines");
						
						if (({ _x in _batteryTypes } count (magazines player)) > 0) then
						{
							_laserBatteriesControl ctrlSetText SPON_RF_LED_ON;
						}
						else
						{
							_laserBatteriesControl ctrlSetText SPON_RF_LED_OFF;
						};
						
						if (not (isNull _target)) then
						{
							_rangeControl ctrlSetText (str (round (player distance _target)));
						};
						
						if (SPON_RF_milsPer360Degrees > 0) then
						{
							private ["_azimuth", "_horizontalAngle",
								"_vector",  "_elevation"];
								
							_vector = player weaponDirection SPON_RF_laserDesignatorClass;
								
							// Azimuth.
							_horizontalAngle = (_vector select 0) atan2 (_vector select 1); // Range: -180 to 180
							_azimuth =  ((_horizontalAngle  + 360) mod 360); // Range: 0 to 360		
							_azimuth = DEGREES_TO_MILS(_azimuth,SPON_RF_milsPer360Degrees);
							_azimuthControl ctrlSetText (str _azimuth);
							
							// Elevation.
							_elevation = asin (_vector select 2);
							_elevation = DEGREES_TO_MILS(_elevation,SPON_RF_milsPer360Degrees);
							if (_elevation >= 0) then
							{
								_elevationControl ctrlSetText (format ["+%1", _elevation]);
							}
							else
							{
								_elevationControl ctrlSetText (str _elevation);
							};
						}
						else
						{
							_azimuthControl ctrlSetText VALUE_NOT_SHOWN;
							_elevationControl ctrlSetText VALUE_NOT_SHOWN;
							
						};
						
						(SPON_RF_display displayCtrl SPON_RF_MODE_LIST_IDC)
							ctrlShow (SPON_RF_milsPer360Degrees > 0);
						
						// waitUntil condition.
						not ((animationState player) in ANIM_BINOCS);
					};
					
					// Clear the overlay, since we are no longer using binoc anim.
					if (not (isNull SPON_RF_display)) then
					{
						cutText ["", "PLAIN", 0];
					};
					
					if (SPON_RF_milsPer360Degrees > 0) then
					{
						call SPON_RF_removeMilsActions;
					};
				};
				
				// waitUntil condition.
				not (player hasWeapon SPON_RF_laserDesignatorClass);
			};
		};
		
		sleep CHECK_FOR_SOFLAM_INVENTORY_INTERVAL;
	};
};

nil; // Return.
