// -----------------------------------------------------------------------------
#define SPON_THIS_FILE SPON\Core\initDebugLog
//
// Copyright (C) 2007 Bil Bas (bil.bagpuss@gmail.com / Spooner)
// License: GNU General Public License, version 3 <http://www.gnu.org/licenses/>  
//
// Last Modified: $Date: 2007/09/21 18:32:35 $
//
// Description:
//   Initialise the debugging log.
//
// Parameters:
//   0: _serverLogging - Whether to log messages on the server and broadcast
//                       then to all players.
//   1: _useAction - Set up an action to open the debug log, rather than key.
//
// Returns:
//   nil
//
// -----------------------------------------------------------------------------

// Ensure the file is not run multiple times or on the server.
if (not (isNil {SPON_debugAppendLog})) exitWith {};

#include "debug.inc.sqf"
#include "macros.inc.sqf"
#include "ui\handles.hpp"

// Global event which broadcasts debug messages from the server.
#define SERVER_DEBUG_EVENT "SPON_serverDebug"

// This is the backtick (`) key.
#define DEBUG_KEYCODE 41

// Select last element of a list box control.
#define SELECT_LAST_ELEMENT(X) lbSetCurSel [X, ((lbSize X) - 1)]

// -----------------------------------------------------------------------------

SPON_GET_PARAMS_2(_serverLogging,_useAction);

// -----------------------------------------------------------------------------

// Most of the log functionality is on the client-side.
if (SPON_isDedicatedServer) then
{
	// -------------------------------------------------------------------------
	// Function: SPON_debugAppendLog (DEDICATED SERVER VERSION)
	//
	// Parameters:
	//   0: _message - Message to append to the log [String]
	//
	// Returns:
	//   nil
	//
	if (_serverLogging) then
	{
		SPON_debugAppendLog =
		{
			SPON_GET_PARAMS_1(_message);
			
			private ["_now", "_logMessage"];
			
			// Prepend the elapsed mission time to the message.
			_now = [time, true] call SPON_formatTime;
			_logMessage = format ["%1 %2", _now, _message];
		
			// The dedicated server broadcasts the log message to the clients.
			if (SPON_serverLogUnqueued) then
			{
				[SERVER_DEBUG_EVENT, [_logMessage]] call SPON_publishGlobalEvent;
			}
			else // There is still an "early" queue to utilise.
			{
				SPON_PUSH(SPON_serverLogQueue,_logMessage);
				
				// Start a thread to send the queue, but only if it doesn't
				// already exist.
				if (isNil "SPON_serverLogQueueSender") then
				{
					SPON_serverLogQueueSender = [] spawn
					{
						// Wait until the timer has started before sending out
						// debug messages from the server to clients.
						waitUntil { time > 0; };
						
						{
							[SERVER_DEBUG_EVENT, [_x]] call SPON_publishGlobalEvent;
						} forEach SPON_serverLogQueue;
					
						// This signals that queue doesn't need to be used any
						// more.
						SPON_serverLogUnqueued = true;
						SPON_serverLogQueue = nil;
						SPON_serverLogQueueSender = nil;
					};
				};
			};
		
			nil; // Return;
		};
		
		// Used by SPON_debugAppendLog (DEDICATED SERVER VERSION) to queue up
		// early messages that wouldn't be picked up by the client.
		SPON_serverLogQueue = [];
		SPON_serverLogUnqueued = false;
	}
	else // not _serverLogging
	{
		// Just make a null function, so debug messages aren't propagated.
		SPON_debugAppendLog = {};
	};
}
else // (not SPON_isDedicatedServer)
{
	// -------------------------------------------------------------------------
	// Function: SPON_debugAppendLog (CLIENT VERSION)
	//
	// Parameters:
	//   0: _message - Message to append to the log [String]
	//
	// Returns:
	//   nil
	//
	// Called from:
	//   Client.
	//
	SPON_debugAppendLog =
	{
		SPON_GET_PARAMS_1(_message);
		
		private ["_now", "_logMessage", "_dialog"];
		
		// Prepend the elapsed mission time to the message.
		_now = [time, true] call SPON_formatTime;
		_logMessage = format ["%1 %2", _now, _message];
	
		// MP-clients (Or SP host) just deal with the messages directly.
		SPON_PUSH(SPON_debugClientLog,_logMessage);
		
		// If the client log is visible, append new message immediately.
		_dialog = findDisplay SPON_DEBUG_DISPLAY_IDD;
		
		if (not (isNull (_dialog displayCtrl SPON_DEBUG_CLIENT_LIST_IDC))) then
		{
			lbAdd [SPON_DEBUG_CLIENT_LIST_IDC, _logMessage];
			
			if (SPON_debugLogUpdates) then
			{
				SELECT_LAST_ELEMENT(SPON_DEBUG_CLIENT_LIST_IDC);
			};
		};

		nil; // Return;
	};
};

// -----------------------------------------------------------------------------

if (SPON_isClient) then
{	
	// -------------------------------------------------------------------------
	// Function: SPON_debugFillLog
	//
	// Description:
	//   Fills the debug log list-boxes with all current messages.
	//
	// Parameters:
	//   None.
	//
	// Returns:
	//   nil
	//
	SPON_debugFillLog =
	{
		// Fill client log.
		lbClear SPON_DEBUG_CLIENT_LIST_IDC;
		{
			lbAdd [SPON_DEBUG_CLIENT_LIST_IDC, _x];
		} forEach SPON_debugClientLog;
		
		if (SPON_debugLogUpdates) then
		{
			SELECT_LAST_ELEMENT(SPON_DEBUG_CLIENT_LIST_IDC);
		};
		
		// Fill server log.
		lbClear SPON_DEBUG_SERVER_LIST_IDC;
		{
			lbAdd [SPON_DEBUG_SERVER_LIST_IDC, _x];
		} forEach SPON_debugServerLog;
		
		if (SPON_debugLogUpdates) then
		{
			SELECT_LAST_ELEMENT(SPON_DEBUG_SERVER_LIST_IDC);
		};
		
		nil; // Return.
	};
	
	// -------------------------------------------------------------------------
	// Function: SPON_debugToggleAutoUpdate
	//
	// Description:
	//  Toggles automatic tracking of the newest message in the log.
	//
	// Parameters:
	//   None.
	//
	// Returns:
	//   nil
	//
	SPON_debugToggleAutoUpdate =
	{
		private ["_text"];
		
		SPON_debugLogUpdates = not SPON_debugLogUpdates;
		
		if (SPON_debugLogUpdates) then
		{
			_text = "On";
		}
		else
		{
			_text = "Off";
		};
		
		ctrlSetText [SPON_DEBUG_UPDATE_STATUS_IDC, _text];
		
		// Push both logs down to the end if auto-updating started.
		if (SPON_debugLogUpdates) then
		{
			SELECT_LAST_ELEMENT(SPON_DEBUG_CLIENT_LIST_IDC);
			SELECT_LAST_ELEMENT(SPON_DEBUG_SERVER_LIST_IDC);
		};
		
		nil; // Return.
	};
	
	// -------------------------------------------------------------------------
	SPON_debugClientLog = [];
	SPON_debugServerLog = [];
	SPON_debugLogUpdates = false;

	// Pick up server logging messages.
	if (SPON_isDedicatedClient and _serverLogging) then
	{
		// Register for server debug events.
		[SERVER_DEBUG_EVENT,
			{
				SPON_GET_PARAMS_1(_logMessage);
				
				//player sideChat (format ["SERVER %1",_logMessage]);
				
				SPON_PUSH(SPON_debugServerLog, _logMessage);
				
				// If the server log is visible, show new messages.
				if (ctrlVisible SPON_DEBUG_SERVER_LIST_IDC) then
				{
					lbAdd [SPON_DEBUG_SERVER_LIST_IDC, _logMessage];
					
					if (SPON_debugLogUpdates) then
					{
						SELECT_LAST_ELEMENT(SPON_DEBUG_SERVER_LIST_IDC);
					};
				};
			}
		] call SPON_addEventHandler;
	};
	
	// Stick an initial message in the server log if it is disabled.
	// TODO: Reformat the dialog so that client-log is larger and server-log
	//       is not visible.
	if (SPON_isDedicatedClient and (not _serverLogging)) then
	{
		SPON_PUSH(SPON_debugServerLog, "Dedicated server logging disabled.");
	}
	else{if (SPON_isClient and isServer) then
	{
		SPON_PUSH(SPON_debugServerLog,
			"Dedicated server logging unavailable in single-player mode.");
	}; };

	// Either add an action to open the debug log or register a key handler.
	if (_useAction) then
	{
		player addAction ["Debug Log", "SPON\Core\viewDebugLog.sqf", nil, 0, false, true];
	}
	else
	{
		// Allow the log to be shown.
		SPON_viewDebugLog = compile preprocessFileLineNumbers "SPON\Core\viewDebugLog.sqf";
			
		["DOWN", DEBUG_KEYCODE,
			{
				SPON_GET_PARAMS_4(_keyCode,_shift,_control,_alt);
			
				private ["_handled"];
				_handled = false;
				
				if ((isNull (findDisplay SPON_DEBUG_DISPLAY_IDD)) and
					(not _shift) and _control and (not _alt)) then
				{
					call SPON_viewDebugLog;
					_handled = true;
				};
				
				_handled; // Return.
			}
		] call SPON_addKeyEventHandler;
	};
};

nil; // Return.