// -----------------------------------------------------------------------------
#define SPON_THIS_FILE SPON\VBIED\initVBIED
//
// Copyright (C) 2007 Bil Bas (bil.bagpuss@gmail.com \ Spooner)
// License: GNU General Public License, version 3 <http://www.gnu.org/licenses/>
//
// Last Modified: $Date: 2007/10/01 17:38:54 $
//
// Description:
//   Initiates a player to place VBIEDs on vehicles.
//
// Parameters:
//   0: _magazineClass - Magazine class to enable as a VBIED, e.g. "PipeBomb",
//        "TimeBomb", etc. [String]
//   1: _explosiveClass - Ammo class to create when exploding a VBIED, e.g.
//        "R_80mm_HE" [String]
//
// Returns:
//   nil
//
// -----------------------------------------------------------------------------

#include "debug.inc.sqf"
#include "macros.inc.sqf"
#include "ieds.inc.sqf"

// -----------------------------------------------------------------------------
// Local functions.
private ["_vehicleFound", "_manFound", "_vbiedDetonated", "_playerKilled"]; 

// -----------------------------------------------------------------------------

_vehicleFound =
{
	SPON_GET_PARAMS_1(_vehicle);
	
	SPON_TRACE_1("Monitoring vehicle",_vehicle);
	
	// Add "Set VBIED" action for new, empty vehicles (if we have satchels).
	if ((({ alive _x } count (crew _vehicle)) == 0) and
		(SPON_vbiedMagazineClass in (magazines player))) then
	{
		[_vehicle] call SPON_addVBIEDAction;
	};
	
	// Disable and re-enable the actions when entering or leaving the vehicle.
	_vehicle addEventHandler ["GETIN",
		{
			SPON_GET_PARAMS_3(_vehicle,_position,_bomber);
			
			if (_bomber == player) then
			{
				private ["_vbiedsAttachedTo"];
		
				// Add "detonate VBIEDs" option, if we have active VBIEDs placed.
				_vbiedsAttachedTo = _bomber getVariable VBIEDS_ATTACHED_TO_VAR;
				if (not (isNil "_vbiedsAttachedTo")) then
				{
					if ((count _vbiedsAttachedTo) > 0) then
					{
						private ["_label"];
						
						if ((count _vbiedsAttachedTo) > 1) then
						{
							_label = format [localize
								"STR_SPON_VBIED_DETONATE_MULTIPLE_ACTION",
								count _vbiedsAttachedTo];
						}
						else
						{
							_label = localize "STR_SPON_VBIED_DETONATE_SINGLE_ACTION";
						};
			
						_action = _vehicle addAction [_label,
							DETONATE_VBIED_ACTION_FILE, nil, 0, false, true];
						_vehicle setVariable [DETONATE_VBIED_IN_VEHICLE_ACTION_VAR,
							_action];
					};
				};
			};
			
			// Always remove the action when someone gets in.
			[_vehicle] call SPON_removeVBIEDAction;
		}
	];
	
	// Getting out of a vehicle, if you are the last one out, will reenable
	// VBIED actions.
	_vehicle addEventHandler ["GETOUT",
		{
			SPON_GET_PARAMS_3(_vehicle,_position,_bomber);
			
			if (_bomber == player) then
			{
				private ["_action"];
				
				// Remove any "detonate VBIEDs" action.
				_action = _vehicle getVariable DETONATE_VBIED_IN_VEHICLE_ACTION_VAR;
				
				if (not (isNil "_action")) then
				{
					_vehicle removeAction _action;
					_vehicle setVariable [DETONATE_VBIED_IN_VEHICLE_ACTION_VAR, nil];
				};
			};
			
			// If no crew are left and player has explosives, replace the action.
			// Note that the player is still a member of the crew when they GETOUT.
			if ((({ alive _x } count (crew _vehicle)) == 1) and
				(SPON_vbiedMagazineClass in (magazines player))) then
			{
				[_vehicle] call SPON_addVBIEDAction;
			};
		}
	];
	
	// When killed, a vehicle can't have anything to do with VBIEDs.
	_vehicle addEventHandler ["KILLED",
		{
			SPON_GET_PARAMS_1(_vehicle);
			
			[_vehicle] call SPON_removeVBIEDAction;
		}
	];
};

// -----------------------------------------------------------------------------
// Description
//   If the man is killed in a vehicle, check to see if it should get its set
//   VBIED action back again.
_manFound =
{
	SPON_GET_PARAMS_1(_man);
	
	_man addEventHandler ["KILLED",
		{
			SPON_GET_PARAMS_1(_deadMan);
			
			SPON_TRACE_1("Man died",_deadMan);
			
			if (SPON_IN_VEHICLE(_deadMan)) then
			{
				if ((({ alive _x } count (crew (vehicle _deadMan))) == 0) and
					(alive (vehicle _deadMan)) and
					(SPON_vbiedMagazineClass in (magazines player))) then
				{
					[vehicle _deadMan] call SPON_addVBIEDAction;
				};
			};
		}
	];
	
	SPON_TRACE_1("Monitoring man",_man);
};

// -----------------------------------------------------------------------------

_vbiedsDetonated =
{
	SPON_GET_PARAMS_1(_numBombs);
	
	SPON_TRACE_1("_vbiedsDetonated",_numBombs);
	
	private ["_i", "_vehicle", "_var"];
	
	for "_i" from 0 to (_numBombs - 1) do
	{
		_var = format ["SPON_vbiedDetonatedIn_%1", _i];
		
		call compile format ["_vehicle = %1", _var];
		
		SPON_TRACE_2("_vbiedsDetonated",_vehicle,crew _vehicle);
		
		if (not (isNull _vehicle)) then
		{
			// Ensure the vehicle is definitely destroyed.
			if (local _vehicle) then
			{
				[_vehicle] spawn
				{
					SPON_GET_PARAMS_1(_vehicle);
					
					// Place an explosive at the centre of the vehicle, then
					// light blue touch paper...
					private ["_pos", "_bomb"];
					
					_pos = _vehicle modelToWorld [0, 0, 0];
					
					_bomb = "Bomb" createVehicle _pos;
					_bomb setPos _pos;
 					_bomb setDammage 1;
 					
 					// Get rid of the exploded bomb model.
					sleep 1;
					deleteVehicle _bomb;
				};
			};
		};
		
		call compile format ["%1 = nil", _var];
	};
};

// -----------------------------------------------------------------------------

_playerKilled =
{
	SPON_GET_PARAMS_1(_deadMan);
	
	private ["_vbiedsAttachedTo"];
	
	_vbiedsAttachedTo = _deadMan getVariable VBIEDS_ATTACHED_TO_VAR;
	
	// Remove any detonate actions.
	if (not (isNil "_vbiedsAttachedTo")) then
	{
		if ((count _vbiedsAttachedTo) > 0) then
		{
			private ["_detonateAction"];
			
			// Check for, and remove, on-foot detonation action.
			_detonateAction = _deadMan getVariable DETONATE_VBIED_ON_FOOT_ACTION_VAR;
			if (not (isNil "_detonateAction")) then
			{
				_deadMan removeAction _detonateAction;
			};
			
			// Check for, and remove, in-vehicle detonation action.
			_detonateAction = _deadMan getVariable
				DETONATE_VBIED_IN_VEHICLE_ACTION_VAR;
			if (not (isNil "_detonateAction")) then
			{
				(vehicle _deadMan) removeAction _detonateAction;
			};
			
			// Replace all the "disarm" actions with "place".
			{
				[_x] call SPON_removeVBIEDAction;
				[_x] call SPON_addVBIEDAction;
			} forEach _vbiedsAttachedTo;
		};
	};
};

// -----------------------------------------------------------------------------

SPON_vbiedGetVehicleName =
{
	SPON_GET_PARAMS_1(_vehicle);
	
	getText (configFile >> "CfgVehicles" >> (typeOf _vehicle) >> "displayName");
};

// -----------------------------------------------------------------------------

SPON_addVBIEDAction =
{
	SPON_GET_PARAMS_1(_vehicle);
	
	private ["_setAction", "_disarmAction"];
	
	// Only add action once, so only add if the vehicle has never had the action
	// or if the action has been removed.
 	_setAction = _vehicle getVariable SET_VBIED_ACTION_VAR;
 	_disarmAction = _vehicle getVariable DISARM_VBIED_ACTION_VAR;

	if ((isNil "_setAction") and (isNil "_disarmAction")) then
	{
		private ["_name", "_action"];
		
		_name = [_vehicle] call SPON_vbiedGetVehicleName;
		_vbiedsAttachedTo = player getVariable VBIEDS_ATTACHED_TO_VAR;
		if (isNil "_vbiedsAttachedTo") then
		{
			_vbiedsAttachedTo = [];
		};
		
		if (_vehicle in _vbiedsAttachedTo) then
		{
			_action = _vehicle addAction [
				format [localize "STR_SPON_VBIED_DISARM_ACTION", _name],
				DISARM_VBIED_ACTION_FILE];
				
			_vehicle setVariable [DISARM_VBIED_ACTION_VAR, _action];
			
			SPON_TRACE_2("SPON_addVBIEDAction - Disarm",_vehicle,_name);
		}
		else // Not set to go off.
		{
			_action = _vehicle addAction [
				format [localize "STR_SPON_VBIED_SET_ACTION", _name],
				SET_VBIED_ACTION_FILE];
				
			_vehicle setVariable [SET_VBIED_ACTION_VAR, _action];
			
			SPON_TRACE_2("SPON_addVBIEDAction - Place",_vehicle,_name);
		};
	};
};

// -----------------------------------------------------------------------------

SPON_removeVBIEDAction =
{
	SPON_GET_PARAMS_1(_vehicle);
	
	private "_setAction";
 	_setAction = _vehicle getVariable SET_VBIED_ACTION_VAR;
 	
	if (not (isNil "_setAction")) then
	{
		_vehicle removeAction _setAction;
		_vehicle setVariable [SET_VBIED_ACTION_VAR, nil];
		SPON_TRACE_1("SPON_removeVBIEDAction - Set",_vehicle);
	}
	else
	{
		private "_disarmAction";
		_disarmAction = _vehicle getVariable DISARM_VBIED_ACTION_VAR;
		
		if (not (isNil "_disarmAction")) then
		{
			_vehicle removeAction _disarmAction;
			_vehicle setVariable [DISARM_VBIED_ACTION_VAR, nil];
			SPON_TRACE_1("SPON_removeVBIEDAction - Disarm",_vehicle);
		};
	};
};

// -----------------------------------------------------------------------------

SPON_vbiedAnimation =
{
	switch ([player] call SPON_MANDO_getPos) do
	{
		case MANDO_GETPOS_STAND:
		{
			_bomber playMove ANIMATION_FROM_STAND;
		};
		case MANDO_GETPOS_CROUCH:
		{
			_bomber playMove ANIMATION_FROM_CROUCH;
		};
	};
};

// -----------------------------------------------------------------------------

SPON_GET_DEFAULT_PARAM(0,_magazineClass,DEFAULT_MAGAZINE_CLASS);
SPON_vbiedMagazineClass = _magazineClass;

SPON_TRACE_1("Initiated",_magazineClass);

// Monitor all new vehicles found.
call SPON_monitorVehicles;

// If any vehicles are already found, then add monitors to all of them.
{
	[_x] call _vehicleFound;
} forEach SPON_vehicles;

["SPON_vehicleFound", _vehicleFound] call SPON_addEventHandler;

["SPON_manFound", _manFound] call SPON_addEventHandler;

// Deal with detonation.
[VBIED_DETONATED_EVENT, _vbiedsDetonated] call SPON_addEventHandler;

SPON_MANDO_getPos = compile preprocessFileLineNumbers
	"SPON\VBIED\thirdParty\MANDO_getPos.sqf";

if (SPON_isClient) then
{
	// Update the attach VBIED actions on all vehicles based on whether the player
	// is carrying any VBIEDs (satchels).
	player addEventHandler ["KILLED", _playerKilled];

	// Monitor the player, to see when he has/hasn't got VBIEDs.
	[] spawn
	{
		while {true} do
		{
			// Wait until we have a VBIED in our inventory.
			while {not (SPON_vbiedMagazineClass in (magazines player))} do
			{
				sleep HAS_VBIED_DELAY;
			};
			
			{
				[_x] call SPON_addVBIEDAction;
			} forEach SPON_vehicles;
	
			// Wait until we have no more VBIEDs left.
			while {SPON_vbiedMagazineClass in (magazines player)} do
			{
				sleep HAS_NO_VBIED_DELAY;
			};
			
			// Remove set/disarm actions from all vehicles.
			{
				[_x] call SPON_removeVBIEDAction;
			} forEach SPON_vehicles;
		};
	};
};

nil; // Return.