/*
	A library to manage regions
	
	At its simplest, a region is just a line that divides the world into two parts.
	In future, a region may be a rectangle, circle or ellipse
	
	Functions:-
	
	region createRegionFromMarker(marker_name, reference point)
	bool insideRegion(region)
	void markRegion(region, objectName)
*/
#include "common.h";

createRegionFromMarker =
{
	private ["_m","_r"];
	_m = ARG0;
	//having found a marker, we can start getting information about it
	_r = [markerPos _m,markerSize _m,markerDir _m,markerShape _m] call createRegion;
	SETVARP(_r,marker,_m);
	_r;
};
/*
	This function creates a region. It calculates all necessary data to treat the
	region as either elliptical or rectangular
*/
createRegion =
{
	private ["_obj","_c1","_f2","_z","_sx","_maj","_size","_pos","_dir","_c2","_sy","_f1","_c4","_c3","_shape"];
	//having found a marker, we can start getting information about it
	_pos = ARG0;
	_size = ARG1;
	_dir = ARG2;
	_shape = ARG3;
	//Work out the four corners of the rectangle by adding and subtracting
	//edge dimensions from the center position. Note that edges are twice
	//the length of the dimensions returned by the 'size' command
	//We treat the rectangle as			c3-----c2
	//						|	|
	//						c4-----c1
	//rotated through the appropriate angle

	//calculate top right by adding sizes to center
	_c2 = [_pos,_size] call addPos;
	//calculate top left by subtracting 2*xsize
	_c3 = [_c2,[-2*EL(_size,0),0]] call addPos;
	//rotate top corners by angle of rectangle
	_c2 = [_c2,_pos,_dir] call rot;
	_c3 = [_c3,_pos,_dir] call rot;
	//obtain bottom corners by flipping top corners around the center
	_c4 = [_pos,_c2] call flipPos;
	_c1 = [_pos,_c3] call flipPos;

	//For an ellipse, work out focal points and major axis...

	_sx = EL(_size,0);
	_sy = EL(_size,1);
	_z= [0,0];
	_maj = 0;
	if (_sy >= _sx) then
	{
		_z = sqrt(_sy*_sy - _sx*_sx);
		_z= [[0,_z],[0,0],_dir] call rot;
		_maj = _sy;
	}
	else
	{
		_z = sqrt(_sx*_sx - _sy*_sy);
		_z= [[_z,0],[0,0],_dir] call rot;
		_maj = _sx;
	};

	_f1 = [_pos,_z] call addPos;
	_f2 = [_pos,_z] call subtractPos;

	//create an object to store information about this border
	_obj = "RoadCone" createVehicle [1,1,1];
	_obj enableSimulation false;
	SETVARP(_obj,center,_pos);
	SETVARP(_obj,sizeX,EL(_size,0));
	SETVARP(_obj,sizeY,EL(_size,1));
	SETVARP(_obj,c1,_c1);
	SETVARP(_obj,c2,_c2);
	SETVARP(_obj,c3,_c3);
	SETVARP(_obj,c4,_c4);
	SETVARP(_obj,dir,_dir);
	SETVARP(_obj,f1,_f1);
	SETVARP(_obj,f2,_f2);
	SETVARP(_obj,majorAxis,_maj);
	SETVARP(_obj,type,_shape);
	SETVARP(_obj,inverted,false);

	//Work out the maximum distance any point in the region can be from the center
	//this is used in various nearestObjects calculations
	if (_shape == "RECTANGLE") then
	{
		//NOTE that the preprocessor can't parse arrays inside macro args
		_size = [0,0] distance _size;
		SETVARP(_obj,maxDistance,_size);
	}
	else
	{
		SETVARP(_obj,maxDistance,_maj);
	};
	//return the region
	_obj;
};
//Draw warning objects around the region
//ARG1 = object type
markRegion =
{
	if (not (ARG1 in ["","none"])) then
	{
		switch (ARG0 getVariable "type" ) do
		{
			case "ELLIPSE":		{_this call markEllipse;};
			case "RECTANGLE":	{_this call markRectangle;};
			case "SEMIPLANE":	{_this call markSemiPlane;};
		};
	};
};
//Will return true if the position in ARG1 is inside the region
insideRegion =
{
	private ["_ret"];
	switch (ARG0 getVariable "type" ) do
	{
		case "ELLIPSE":		{_ret = _this call insideEllipse;};
		case "RECTANGLE":	{_ret = _this call insideRectangle;};
		case "SEMIPLANE":	{_ret = _this call insideSemiPlane;};
	};
	if (VAR(ARG0,inverted)) then {_ret = not _ret;};
	_ret;
};
//wrapper function that only bothers if object is on ground...
insideRegionOnGround =
{
	private ["_p","_ret"];
	_p = ARG1;
	if (count _p < 3) then {_p set [2,0];};
	if (EL(_p,2) <5) then
	{
		_ret = _this call insideRegion;
	}
	else
	{
		_ret = false;
	};
	_ret;
};
randPosInsideRegion =
{
	private ["_r","_c","_d","_p","_done"];
	_r = ARG0;
	_c = _r getVariable "center";
	_d = _r getVariable "majorAxis";
	_done = false;
	while {not _done} do
	{
		_p = [_c,0,_d] call randPos;
		_done = [_r,_p] call insideRegion;
	};
	_p;
};
/*********************************************************************
	RECTANGULAR REGION CODE
*/
//create objects every 10 meters along a line
markLine =
{
	private ["_p2","_nc","_rc","_c","_len","_objType","_dir","_p1","_v"];

	_p1 = ARG0;
	_p2 = ARG1;
	_objType = ARG2;
	_dir = ARG3;
	_len = _p1 distance _p2;

	_nc = (floor _len) /10;
	for "_c" from 1 to _nc do
	{
		_rc =
		[
			EL(_p1,0) + (EL(_p2,0) - EL(_p1,0))*_c/_nc,
			EL(_p1,1) + (EL(_p2,1) - EL(_p1,1))*_c/_nc
		];
		if (isServer) then
		{
			_v = _objType createVehicle _rc;
		}
		else
		{
			_v = _objType createVehicleLocal _rc;
		};
		_v setDir _dir;
		_v setPos _rc;
		_v enableSimulation false;
	};
};
markRectangle =
{
	private ["_region","_objType","_c","_b","_d","_a"];
	_region = _this select 0;
	_objType = _this select 1;

	_c = _region getVariable "center";

	_a = _region getVariable "c1";
	_b = _region getVariable "c2";
	_d = [ [_a,_b] call midpoint,_c] call bearing;
	if (VAR(_region,inverted)) then {_d = _d + 180;};

	[_a,_b,_objType,_d ] call markLine;

	_a = _region getVariable "c3";
	_d = [ [_a,_b] call midpoint,_c] call bearing;
	if (VAR(_region,inverted)) then {_d = _d + 180;};
	[_a,_b,_objType,_d ] call markLine;

	_b = _region getVariable "c4";
	_d = [ [_a,_b] call midpoint,_c] call bearing;
	if (VAR(_region,inverted)) then {_d = _d + 180;};
	[_a,_b,_objType,_d ] call markLine;

	_a = _region getVariable "c1";
	_d = [ [_a,_b] call midpoint,_c] call bearing;
	if (VAR(_region,inverted)) then {_d = _d + 180;};
	[_a,_b,_objType,_d ] call markLine;
};
insideRectangle =
{
	private ["_region","_sx","_c","_pos","_dir","_sy"];
	_region = _this select 0;
	_pos = _this select 1;
	_c = _region getVariable "center";
	_dir = _region getVariable "dir";
	_dir= - _dir;

	_pos = [_pos,_c,_dir] call rot;
	_pos = [EL(_pos,0)-EL(_c,0),EL(_pos,1)-EL(_c,1)];
	_sx = _region getVariable "sizeX";
	_sy = _region getVariable "sizeY";

	if (( abs(EL(_pos,0)) <= _sx) && ( abs(EL(_pos,1)) <=_sy)) then {true;} else {false;};
};

/*********************************************************************
	ELIPTICAL REGION CODE
*/
markEllipse =
{
	private ["_bear","_region","_sx","_objType","_c","_obj","_pos","_dir","_sy","_i"];
	_region = _this select 0;
	_objType = _this select 1;
	// Place objects along the border every 3 meters
	_c = _region getVariable "center";
	_sx = _region getVariable "sizeX";
	_sy = _region getVariable "sizeY";
	_dir = _region getVariable "dir";
	for "_i" from 0 to 360 do
	{
		_pos = [ (_sx * sin(_i)) + EL(_c,0),
			 (_sy * cos(_i)) + EL(_c,1)];
		_pos = [_pos,_c,_dir] call rot;
		if (isServer) then
		{
			_obj = _objType createVehicle _pos;
		}
		else
		{
			_obj = _objType createVehicleLocal _pos;
		};
		_obj setPos _pos;
		_obj enableSimulation false;
		_bear = [_pos,_c] call bearing;
		if (VAR(_region,inverted)) then {_bear = _bear + 180;};
		_obj setDir _bear;
		_i = _i + 10;
	};
};
insideEllipse =
{
	private ["_region","_f2","_pos","_f1","_d"];
	_region = _this select 0;
	_pos = _this select 1;
	_f1 = _region getVariable "f1";
	_f2 = _region getVariable "f2";
	_d = _region getVariable "majorAxis";

	if ( ((_pos distance _f1) + (_pos distance _f2)) <= (2* _d)) then {true;} else {false;};
};

/*********************************************************************
	OBSOLETE SEMI-PLANE region code
*/
makeRegionSemiPlane =
{
	private ["_region","_r","_cp","_dy","_c1","_d1","_s","_p2","_ref","_p1","_d2","_c2","_c4","_dx","_c3"];
	_region = ARG0;
	_ref = ARG1;
	//having found a marker, we can start getting information about it
	_c1 = _region getVariable "c1";
	_c2 = _region getVariable "c2";
	_c3 = _region getVariable "c3";
	_c4 = _region getVariable "c4";
	//choose the two longest sides
	_s = [];
	if ((_region getVariable "sizeY") > (_region getVariable "sizeX")) then
	{
		_s = [_c1,_c2,_c4,_c3];
	}
	else
	{
		_s = [_c2,_c3,_c1,_c4];
	};
	//Of the two longest sides, choose the nearest side to the reference point
	//and store the vertices of the nearest side in the '_r' array
	_d1 = _ref distance ([EL(_s,0),EL(_s,1)] call midpoint);
	_d2 = _ref distance ([EL(_s,2),EL(_s,3)] call midpoint);

	_r = [];
	if (_d1 < _d2) then
	{
		_r = [EL(_s,0),EL(_s,1)];
	}
	else
	{
		_r = [EL(_s,2),EL(_s,3)]
	};
	//Precompute information for future cross-products
	_p1 = EL(_r,0);
	_p2 = EL(_r,1);
	_dx = EL(_p2,0) - EL(_p1,0);
	_dy = EL(_p2,1) - EL(_p1,1);
	//work out the cross-product of the reference point so we know which
	//side of the line we should be
	_cp = ((EL(_ref,1) - EL(_p1,1)) * _dx) - ((EL(_ref,0) - EL(_p1,0))* _dy);
	_cp = _cp / (abs _cp);

	SETVARP(_region,a,_p1);
	SETVARP(_region,b,_p2);
	SETVARP(_region,dx,_dx);
	SETVARP(_region,dy,_dy);
	SETVARP(_region,cpsign,_cp);
	SETVARP(_region,type,"SEMIPLANE");
};
//mark this region as inverted - ie it is inside-out
makeRegionInverted =
{
	SETVARP(ARG0,inverted,true);
};
//create objects all along the region border
markSemiPlane =
{
	private ["_region","_objType","_c","_b","_a","_dir"];
	_region = ARG0;
	_objType = ARG1;
	_a = _region getVariable "a";
	_b = _region getVariable "b";
	_c = _region getVariable "center";
	_dir = [[_a,_b] call midPoint,_c] call bearing;

	[_a,_b,_objType,_dir] call markLine;
};
//return TRUE if the supplied position is inside the region
insideSemiPlane =
{
	private ["_cp","_dy","_region","_pos","_dx","_a"];
	_region = _this select 0;
	_pos = _this select 1;
	_dx = _region getVariable "dx";
	_dy = _region getVariable "dy";
	_a = _region getVariable "a";
	_cp = ((EL(_pos,1) - EL(_a,1)) * _dx) - ((EL(_pos,0) - EL(_a,0))* _dy);
	_cp = _cp / (abs _cp);
	_cp = _cp * (_region getVariable "cpsign" );
	if (_cp <0) then {true;} else {false;};
};
//returns a list of objects of the requested class inside a region
objectsInsideRegion =
{
	private ["_r","_class","_ret"];
	_r = ARG0;
	_class = ARG1;
	_ret = [];
	{
		if ([_r,position _x] call insideRegion) then
		{
			_ret = _ret +[_x];
		};
	} forEach nearestObjects [VAR(_r,center),[_class],VAR(_r,maxDistance)];
	_ret;
};
nearestEdgePoint =
{
	private ["_r","_p","_x","_y"];
	_r = ARG0;
	_p = ARG1;
	VAR(_r,dir);
	_p = [_p,VAR(_r,center)] call subtractPos;
	_p = [_p,[0,0],-1* VAR(_r,dir)] call rot;
	_x = EL(_p,0);
	_y = EL(_p,1);
	_x = (VAR(_r,sizeX) - abs _x);
	_y = (VAR(_r,sizeY) - abs _y);

	if (_x > _y) then
	{
		_p set [1,VAR(_r,sizeY) * SIGN(EL(_p,1))];
	}
	else
	{
		_p set [0,VAR(_r,sizeX) * SIGN(EL(_p,0))];
	};
	_p = [_p,[0,0],VAR(_r,dir)] call rot;
	_p = [_p,VAR(_r,center)] call addPos;
	_p;
};